/*
 * Decompiled with CFR 0.152.
 */
package org.rvpf.jnlp.loader;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;
import org.rvpf.base.DateTime;
import org.rvpf.base.ElapsedTime;
import org.rvpf.base.logger.Logger;
import org.rvpf.base.logger.Message;
import org.rvpf.base.logger.Messages;
import org.rvpf.base.tool.Require;
import org.rvpf.base.util.container.IdentityHashSet;
import org.rvpf.jnlp.loader.JNLPException;
import org.rvpf.jnlp.loader.JNLPMessages;
import org.rvpf.jnlp.loader.JNLPProperties;

@NotThreadSafe
public final class CacheManager {
    public static final String CACHE_DIR_PROPERTY = "cache.dir";
    public static final String CACHE_PREFIX_PROPERTY = "cache.prefix";
    public static final String DEFAULT_CACHE_DIR = "cache";
    public static final String DEFAULT_CACHE_PREFIX = "jnlp-";
    public static final int DEFAULT_RETRY_DELAY = 60;
    public static final String INDEX_FILE_NAME = "index.properties";
    public static final String INSIST_PROPERTY = "download.insist";
    public static final String LOCK_FILE_NAME = "cache.lock";
    public static final ElapsedTime MINIMUM_RETRY_DELAY = ElapsedTime.fromMillis(1000L);
    public static final String PURGE_PROPERTY = "cache.purge";
    public static final String RETRIES_PROPERTY = "download.retries";
    public static final String RETRY_DELAY_PROPERTY = "download.retry.delay";
    private static final Logger _LOGGER = Logger.getInstance(CacheManager.class);
    private static final int _STREAM_BUFFER_SIZE = 8192;
    private final File _cacheDir;
    private final Map<String, File> _cacheIndex = new LinkedHashMap<String, File>();
    private boolean _connectionFailed;
    private Boolean _insist;
    private final RandomAccessFile _lockFile;
    private DateTime _nameStamp = DateTime.fromRaw(0L);
    private final Set<File> _newEntries = new IdentityHashSet<File>();
    private final String _prefix = JNLPProperties.getInstance().getStringValue("cache.prefix", Optional.of("jnlp-")).get();
    private int _retried;
    private int _retries;
    private ElapsedTime _retryDelay;

    CacheManager() throws IOException {
        this._cacheDir = new File(JNLPProperties.getInstance().getStringValue(CACHE_DIR_PROPERTY, Optional.of(DEFAULT_CACHE_DIR)).get());
        if (this._cacheDir.mkdirs()) {
            _LOGGER.info(JNLPMessages.CACHE_CREATED, this._cacheDir.getAbsolutePath());
        } else if (this._cacheDir.isDirectory()) {
            _LOGGER.debug(JNLPMessages.CACHE_DIRECTORY, this._cacheDir.getAbsolutePath());
        } else {
            throw new IOException(Message.format(JNLPMessages.CACHE_FAILED, this._cacheDir.getAbsolutePath()));
        }
        File lockFile = new File(this._cacheDir, this._prefix + LOCK_FILE_NAME);
        this._lockFile = new RandomAccessFile(lockFile, "rw");
        this._lockFile.getChannel().lock();
        _LOGGER.trace(JNLPMessages.CACHE_LOCKED, new Object[0]);
        this._purge();
        this._loadIndex();
        this._cleanUp();
    }

    void close() throws IOException {
        this._lockFile.close();
        _LOGGER.trace(JNLPMessages.CACHE_UNLOCKED, new Object[0]);
    }

    @Nonnull
    @CheckReturnValue
    File getFile(@Nonnull URL url) throws Exception {
        File cachedFile;
        block10: {
            long after;
            block11: {
                File newCachedFile;
                long lastModified;
                _LOGGER.trace(JNLPMessages.LOOKING_FOR, url);
                cachedFile = this._cacheIndex.get(url.toExternalForm());
                if (this._connectionFailed && cachedFile != null) {
                    return cachedFile;
                }
                after = cachedFile != null ? cachedFile.lastModified() : 0L;
                this._retried = 0;
                while (true) {
                    URLConnection connection;
                    if ((connection = this._openConnection(url, after)) == null) {
                        if ((cachedFile == null || this._insist()) && this._retry()) {
                            continue;
                        }
                        break block10;
                    }
                    lastModified = connection.getLastModified();
                    if (lastModified <= after) break block11;
                    try {
                        newCachedFile = this._download(connection);
                    }
                    catch (IOException exception) {
                        if ((cachedFile == null || this._insist()) && this._retry()) {
                            continue;
                        }
                        break block10;
                    }
                    break;
                }
                if (!newCachedFile.setLastModified((lastModified + 999L) / 1000L * 1000L)) {
                    throw new JNLPException(Message.format(JNLPMessages.SET_FILE_MODIFIED_FAILED, newCachedFile));
                }
                this._cacheIndex.put(url.toExternalForm(), newCachedFile);
                this._saveIndex();
                this._newEntries.add(newCachedFile);
                CacheManager._deleteObsoleteFile(cachedFile, url);
                return newCachedFile;
            }
            if (after == 0L) {
                throw new JNLPException(Message.format(JNLPMessages.LAST_MODIFIED_MISSING, new Object[0]));
            }
            Require.notNull((Object)cachedFile);
            _LOGGER.trace(JNLPMessages.FILE_UP_TO_DATE, cachedFile.getName());
        }
        if (cachedFile == null) {
            throw new JNLPException(Message.format(JNLPMessages.GET_FILE_FAILED, new Object[0]));
        }
        return cachedFile;
    }

    @CheckReturnValue
    boolean isNew(@Nonnull File cachedFile) {
        return this._newEntries.contains(cachedFile);
    }

    private static void _deleteObsoleteFile(File cachedFile, URL url) {
        if (cachedFile != null) {
            if (cachedFile.delete()) {
                _LOGGER.trace(JNLPMessages.OBSOLETE_REMOVED, cachedFile.getName(), url);
            } else {
                _LOGGER.warn(JNLPMessages.OBSOLETE_REMOVE_FAILED, cachedFile.getName(), url);
            }
        }
    }

    private void _cleanUp() {
        File[] cacheDirFiles = (File[])Require.notNull((Object)this._cacheDir.listFiles());
        HashSet<File> cachedFiles = new HashSet<File>(this._cacheIndex.values());
        for (File file : cacheDirFiles) {
            if (!file.isFile() || !file.getName().startsWith(this._prefix) || cachedFiles.contains(file) || file.getName().endsWith(LOCK_FILE_NAME) || file.getName().endsWith(INDEX_FILE_NAME)) continue;
            if (file.delete()) {
                _LOGGER.trace(JNLPMessages.UNREF_REMOVED, file.getName());
                continue;
            }
            _LOGGER.warn(JNLPMessages.UNREF_REMOVE_FAILED, file.getName());
        }
    }

    private URLConnection _connectionFailed(Messages.Entry entry, Object ... params) {
        this._connectionFailed = true;
        if (this._retried == 0) {
            _LOGGER.warn(entry, params);
        }
        return null;
    }

    private File _download(URLConnection connection) throws IOException {
        DateTime nameStamp = DateTime.now();
        if (nameStamp.isNotAfter(this._nameStamp)) {
            nameStamp = this._nameStamp.after();
        }
        this._nameStamp = nameStamp;
        File cachedFile = new File(this._cacheDir, this._prefix + this._nameStamp.toFileName());
        _LOGGER.debug(JNLPMessages.DOWNLOADING_FROM, cachedFile.getName(), connection.getURL());
        InputStream connectionStream = connection.getInputStream();
        FileOutputStream fileStream = new FileOutputStream(cachedFile);
        byte[] buffer = new byte[8192];
        try {
            int length;
            while ((length = connectionStream.read(buffer)) >= 0) {
                ((OutputStream)fileStream).write(buffer, 0, length);
            }
        }
        catch (IOException exception) {
            _LOGGER.warn(JNLPMessages.DOWNLOAD_FAILED, exception.getMessage());
            ((OutputStream)fileStream).close();
            if (!cachedFile.delete()) {
                _LOGGER.warn(JNLPMessages.DOWNLOAD_DELETE_FAILED, cachedFile.getName());
            }
            throw exception;
        }
        finally {
            connectionStream.close();
            ((OutputStream)fileStream).close();
        }
        return cachedFile;
    }

    private boolean _insist() {
        if (this._insist == null) {
            this._insist = JNLPProperties.getInstance().getBooleanValue(INSIST_PROPERTY);
        }
        return this._insist;
    }

    private void _loadIndex() throws IOException {
        File indexFile = new File(this._cacheDir, this._prefix + INDEX_FILE_NAME);
        if (indexFile.exists()) {
            _LOGGER.trace(JNLPMessages.CACHE_INDEX_FOUND, indexFile.getName());
            Properties properties = new Properties();
            try (BufferedInputStream indexStream = new BufferedInputStream(new FileInputStream(indexFile));){
                properties.load(indexStream);
            }
            for (Map.Entry<Object, Object> entry : properties.entrySet()) {
                File cachedFile = new File(this._cacheDir, this._prefix + entry.getKey());
                DateTime nameStamp = DateTime.fromString((String)entry.getKey());
                URL url = new URL((String)entry.getValue());
                if (nameStamp.isAfter(this._nameStamp)) {
                    this._nameStamp = nameStamp;
                }
                if (cachedFile.isFile()) {
                    _LOGGER.trace(JNLPMessages.CACHE_FILE_FOUND, cachedFile.getName(), url);
                    this._cacheIndex.put(url.toExternalForm(), cachedFile);
                    continue;
                }
                _LOGGER.warn(JNLPMessages.CACHE_FILE_LOST, cachedFile.getName());
            }
        }
    }

    private URLConnection _openConnection(URL url, long after) throws Exception {
        URLConnection connection = url.openConnection();
        connection.setUseCaches(false);
        if (after > 0L) {
            connection.setIfModifiedSince(after);
        }
        try {
            connection.connect();
        }
        catch (IOException exception) {
            return this._connectionFailed(JNLPMessages.CONNECT_FAILED, url, exception.getMessage());
        }
        if (connection instanceof HttpURLConnection) {
            int responseCode;
            try {
                responseCode = ((HttpURLConnection)connection).getResponseCode();
            }
            catch (IOException exception) {
                return this._connectionFailed(JNLPMessages.RESPONSE_FAILED, url, exception.getMessage());
            }
            if (responseCode != 200 && responseCode != 304) {
                return this._connectionFailed(JNLPMessages.RESPONSE_UNEXPECTED, url, String.valueOf(responseCode), URLDecoder.decode(((HttpURLConnection)connection).getResponseMessage(), StandardCharsets.UTF_8.name()));
            }
            if (this._retried > 0) {
                _LOGGER.info(JNLPMessages.CONNECT_SUCCEEDED, url);
                this._connectionFailed = false;
            }
        }
        return connection;
    }

    private void _purge() {
        if (JNLPProperties.getInstance().getBooleanValue(PURGE_PROPERTY)) {
            File[] cacheDirFiles;
            for (File cacheDirFile : cacheDirFiles = (File[])Require.notNull((Object)this._cacheDir.listFiles())) {
                if (!cacheDirFile.isFile() || !cacheDirFile.getName().startsWith(this._prefix) || cacheDirFile.getName().endsWith(LOCK_FILE_NAME)) continue;
                if (cacheDirFile.delete()) {
                    _LOGGER.trace(JNLPMessages.PURGED, cacheDirFile.getName());
                    continue;
                }
                _LOGGER.warn(JNLPMessages.PURGE_FAILED, cacheDirFile.getName());
            }
        }
    }

    private boolean _retry() throws InterruptedException {
        if (this._retryDelay == null) {
            ElapsedTime defaultRetryDelay = ElapsedTime.fromMillis(60000L);
            this._retries = JNLPProperties.getInstance().getIntValue(RETRIES_PROPERTY, -1);
            this._retryDelay = JNLPProperties.getInstance().getElapsedValue(RETRY_DELAY_PROPERTY, Optional.of(defaultRetryDelay), Optional.of(defaultRetryDelay)).get();
            if (this._retryDelay.compareTo(MINIMUM_RETRY_DELAY) < 0) {
                this._retryDelay = defaultRetryDelay;
            }
        }
        if (this._retries >= 0 && this._retried >= this._retries) {
            return false;
        }
        if (this._retried == 0) {
            _LOGGER.info(JNLPMessages.WILL_RETRY, this._retries, this._retryDelay);
        }
        Thread.sleep(this._retryDelay.toMillis());
        ++this._retried;
        return true;
    }

    private void _saveIndex() throws IOException {
        Properties properties = new Properties();
        for (Map.Entry<String, File> entry : this._cacheIndex.entrySet()) {
            properties.setProperty(entry.getValue().getName().substring(this._prefix.length()), entry.getKey());
        }
        File indexFile = new File(this._cacheDir, this._prefix + INDEX_FILE_NAME);
        try (BufferedOutputStream indexStream = new BufferedOutputStream(new FileOutputStream(indexFile));){
            properties.store(indexStream, "Cache index.");
        }
    }
}

