/*
 * Decompiled with CFR 0.152.
 */
package org.rvpf.pap.modbus.transport;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ProtocolException;
import java.net.Socket;
import java.net.SocketException;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import org.rvpf.base.ElapsedTime;
import org.rvpf.base.logger.Logger;
import org.rvpf.base.logger.Messages;
import org.rvpf.base.tool.Require;
import org.rvpf.base.tool.Traces;
import org.rvpf.pap.SerialPortInputStream;
import org.rvpf.pap.SerialPortOutputStream;
import org.rvpf.pap.SerialPortWrapper;
import org.rvpf.pap.TraceBuffer;
import org.rvpf.pap.modbus.ModbusMessages;
import org.rvpf.pap.modbus.message.Prefix;
import org.rvpf.pap.modbus.transport.CRC;
import org.rvpf.pap.modbus.transport.LRC;

public abstract class Transport
implements Closeable {
    public static final int SERIAL_MODE_ASCII = 2;
    public static final int SERIAL_MODE_ASCII_DATA_BITS = 7;
    public static final String SERIAL_MODE_ASCII_NAME = "ASCII";
    public static final int SERIAL_MODE_RTU = 1;
    public static final int SERIAL_MODE_RTU_DATA_BITS = 8;
    public static final String SERIAL_MODE_RTU_NAME = "RTU";
    private boolean _littleEndian;
    private final Logger _logger = Logger.getInstance(this.getClass());
    private InputStream _receiveStream;
    private long _receiveTimeout;
    private final TraceBuffer _receivedBytes;
    private OutputStream _sendStream;
    private final TraceBuffer _sentBytes;
    private final Traces _traces;
    private final byte _unitIdentifier;

    protected Transport(byte unitIdentifier, @Nonnull Traces traces) {
        this._unitIdentifier = unitIdentifier;
        this._traces = traces;
        this._receivedBytes = new TraceBuffer(this._traces.isEnabled());
        this._sentBytes = new TraceBuffer(this._traces.isEnabled());
    }

    @Nonnull
    @CheckReturnValue
    public static Transport newSerialPortTransport(@Nonnull SerialPortWrapper serialPort, int serialMode, byte unitIdentifier, @Nonnull Traces traces) {
        switch (serialMode) {
            case 1: {
                return new _OnSerialRTUPort(serialPort, unitIdentifier, traces);
            }
            case 2: {
                return new _OnSerialASCIIPort(serialPort, unitIdentifier, traces);
            }
        }
        throw new InternalError();
    }

    @Nonnull
    @CheckReturnValue
    public static Transport newSocketTransport(@Nonnull Socket socket, byte unitIdentifier, @Nonnull Traces traces) {
        return new _OnSocket(socket, unitIdentifier, traces);
    }

    @CheckReturnValue
    public static int serialMode(String serialModeName) {
        if (serialModeName == null || serialModeName.isEmpty()) {
            return 1;
        }
        switch (serialModeName.toUpperCase(Locale.ROOT)) {
            case "RTU": {
                return 1;
            }
            case "ASCII": {
                return 2;
            }
        }
        Logger.getInstance(Transport.class).warn((Messages.Entry)ModbusMessages.UNRECOGNIZED_SERIAL_MODE, new Object[]{serialModeName});
        return 1;
    }

    @CheckReturnValue
    public static int serialModeDataBits(int serialMode) {
        switch (serialMode) {
            case 1: {
                return 8;
            }
            case 2: {
                return 7;
            }
        }
        throw new InternalError();
    }

    @Nonnull
    @CheckReturnValue
    public static String serialModeName(int serialMode) {
        switch (serialMode) {
            case 1: {
                return SERIAL_MODE_RTU_NAME;
            }
            case 2: {
                return SERIAL_MODE_ASCII_NAME;
            }
        }
        throw new InternalError();
    }

    @Override
    public void close() throws IOException {
        if (!this._sentBytes.isEmpty()) {
            this.traceSentBytes();
        }
        if (!this._receivedBytes.isEmpty()) {
            this.traceReceivedBytes();
        }
        this.trace("End");
        if (this._traces.isEnabled()) {
            this._traces.commit();
            this._traces.tearDown();
        }
    }

    public final void flush() throws IOException {
        this._sendStream.flush();
    }

    @CheckReturnValue
    public abstract int getBatchSize();

    @Nonnull
    @CheckReturnValue
    public abstract String getLocalAddress();

    @Nonnull
    @CheckReturnValue
    public abstract String getRemoteAddress();

    @CheckReturnValue
    public byte getUnitIdentifier() {
        return this._unitIdentifier;
    }

    @Nonnull
    @CheckReturnValue
    public abstract Prefix newPrefix();

    public abstract void onMessageReceiveCompleted() throws InterruptedException;

    public abstract void onMessageSendBegins() throws IOException;

    @CheckReturnValue
    public byte receiveByte() throws EOFException, IOException {
        int read = this._receiveStream.read();
        if (read < 0) {
            throw new EOFException();
        }
        this._receivedBytes.append((byte)read);
        return (byte)read;
    }

    @Nonnull
    @CheckReturnValue
    public abstract Prefix receivePrefix() throws EOFException, IOException;

    @CheckReturnValue
    public final short receiveShort() throws EOFException, IOException {
        byte read1 = this.receiveByte();
        byte read2 = this.receiveByte();
        return (short)(this._littleEndian ? read1 & 0xFF | read2 << 8 : read1 << 8 | read2 & 0xFF);
    }

    @CheckReturnValue
    public abstract boolean receiveSuffix() throws EOFException, IOException;

    public void sendByte(int byteValue) throws IOException {
        this._sendStream.write(byteValue);
        this._sentBytes.append((byte)byteValue);
    }

    public abstract void sendPrefix(@Nonnull Prefix var1) throws IOException;

    public final void sendShort(int shortValue) throws IOException {
        if (this._littleEndian) {
            this.sendByte(shortValue);
            this.sendByte(shortValue >> 8);
        } else {
            this.sendByte(shortValue >> 8);
            this.sendByte(shortValue);
        }
    }

    public abstract void sendSuffix() throws IOException;

    public abstract void setBatchSize(int var1);

    public final void setLittleEndian(boolean littleEndian) {
        this._littleEndian = littleEndian;
    }

    public final void setReceiveTimeout(@Nonnull Optional<ElapsedTime> receiveTimeout) {
        this._receiveTimeout = receiveTimeout.isPresent() ? receiveTimeout.get().toMillis() : -1L;
    }

    @Nonnull
    @CheckReturnValue
    protected static String byteToHexString(byte byteValue) {
        StringBuilder builder = new StringBuilder();
        builder.append((char)Transport.halfByteToHexDigit((byte)(byteValue >> 4)));
        builder.append((char)Transport.halfByteToHexDigit(byteValue));
        return builder.toString();
    }

    @CheckReturnValue
    protected static byte halfByteToHexDigit(byte halfByte) {
        byte hexDigit = (halfByte = (byte)(halfByte & 0xF)) < 10 ? (byte)(48 + halfByte) : (byte)(65 + halfByte - 10);
        return hexDigit;
    }

    @CheckReturnValue
    protected static byte hexDigitToHalfByte(byte hexDigit) throws ProtocolException {
        if (48 <= hexDigit && hexDigit <= 57) {
            return (byte)(hexDigit - 48);
        }
        if (65 <= hexDigit && hexDigit <= 70) {
            return (byte)(10 + hexDigit - 65);
        }
        if (97 <= hexDigit && hexDigit <= 102) {
            return (byte)(10 + hexDigit - 97);
        }
        throw new ProtocolException();
    }

    @CheckReturnValue
    protected long getReceiveTimeout() {
        return this._receiveTimeout;
    }

    @Nonnull
    @CheckReturnValue
    protected TraceBuffer getReceivedBytes() {
        return this._receivedBytes;
    }

    @Nonnull
    @CheckReturnValue
    protected TraceBuffer getSentBytes() {
        return this._sentBytes;
    }

    @Nonnull
    @CheckReturnValue
    protected Logger getThisLogger() {
        return this._logger;
    }

    protected final void setReceiveStream(@Nonnull InputStream receiveStream) {
        this._receiveStream = new BufferedInputStream(receiveStream);
    }

    protected final void setSendStream(@Nonnull OutputStream sendStream) {
        this._sendStream = new BufferedOutputStream(sendStream);
    }

    protected final void trace(@Nonnull String message) {
        if (this._traces.isEnabled()) {
            this._traces.add(Optional.of(this.getRemoteAddress()), (Object)message);
            this._traces.commit();
        }
    }

    protected final void traceReceivedBytes() {
        if (this._traces.isEnabled()) {
            this.trace("Received: " + this._receivedBytes);
            this._receivedBytes.reset();
        }
    }

    protected final void traceSentBytes() {
        if (this._traces.isEnabled()) {
            this.trace("Sent: " + this._sentBytes);
            this._sentBytes.reset();
        }
    }

    private static final class _OnSocket
    extends Transport {
        private int _batchSize;
        private final AtomicInteger _nextTransaction = new AtomicInteger();
        private final Socket _socket;

        _OnSocket(@Nonnull Socket socket, byte unitIdentifier, @Nonnull Traces traces) {
            super(unitIdentifier, traces);
            this._socket = socket;
            try {
                this.setReceiveStream(this._socket.getInputStream());
                this.setSendStream(this._socket.getOutputStream());
            }
            catch (IOException exception) {
                throw new RuntimeException(exception);
            }
            this.trace("Begin");
        }

        @Override
        public void close() throws IOException {
            try {
                this._socket.close();
            }
            finally {
                super.close();
            }
        }

        @Override
        public int getBatchSize() {
            return this._batchSize;
        }

        @Override
        public String getLocalAddress() {
            return this._socket.getLocalSocketAddress().toString();
        }

        @Override
        public String getRemoteAddress() {
            return this._socket.getRemoteSocketAddress().toString();
        }

        @Override
        public Prefix newPrefix() {
            return new Prefix.MBAP((short)this._nextTransaction.getAndIncrement(), this.getUnitIdentifier());
        }

        @Override
        public void onMessageReceiveCompleted() {
        }

        @Override
        public void onMessageSendBegins() {
        }

        @Override
        public byte receiveByte() throws EOFException, IOException {
            try {
                return super.receiveByte();
            }
            catch (SocketException exception) {
                throw this._socket.isClosed() ? new EOFException() : exception;
            }
        }

        @Override
        public Prefix receivePrefix() throws EOFException, IOException {
            this.getReceivedBytes().reset();
            return Prefix.MBAP.read(this);
        }

        @Override
        public boolean receiveSuffix() {
            this.traceReceivedBytes();
            return true;
        }

        @Override
        public void sendPrefix(Prefix prefix) throws IOException {
            prefix.write(this);
        }

        @Override
        public void sendSuffix() {
            this.traceSentBytes();
        }

        @Override
        public void setBatchSize(int batchSize) {
            this._batchSize = batchSize;
        }
    }

    private static final class _OnSerialRTUPort
    extends _OnSerialPort {
        private final CRC _inputCRC = new CRC();
        private final CRC _outputCRC = new CRC();

        _OnSerialRTUPort(@Nonnull SerialPortWrapper serialPort, byte unitIdentifier, @Nonnull Traces traces) {
            super(serialPort, unitIdentifier, traces);
            Require.equal((long)8L, (long)serialPort.getPortDataBits());
        }

        @Override
        public byte receiveByte() throws EOFException, IOException {
            byte readByte = super.receiveByte();
            this._inputCRC.update(readByte);
            return readByte;
        }

        @Override
        public Prefix receivePrefix() throws EOFException, IOException {
            this._inputCRC.reset();
            this.getReceivedBytes().reset();
            return new Prefix(this.receiveByte());
        }

        @Override
        public boolean receiveSuffix() throws EOFException, IOException {
            byte crcHigh;
            long crc = this._inputCRC.getValue();
            byte crcLow = super.receiveByte();
            boolean success = (long)(crcLow & 0xFF | ((crcHigh = super.receiveByte()) & 0xFF) << 8) == crc;
            this.traceReceivedBytes();
            return success;
        }

        @Override
        public void sendByte(int byteValue) throws IOException {
            super.sendByte(byteValue);
            this._outputCRC.update(byteValue);
        }

        @Override
        public void sendPrefix(Prefix prefix) throws IOException {
            this._outputCRC.reset();
            prefix.write(this);
        }

        @Override
        public void sendSuffix() throws IOException {
            int crc = (int)this._outputCRC.getValue();
            super.sendByte(crc);
            super.sendByte(crc >> 8);
            this.traceSentBytes();
        }
    }

    private static abstract class _OnSerialPort
    extends Transport
    implements SerialPortWrapper.StatusChangeListener {
        private final SerialPortWrapper _serialPort;

        _OnSerialPort(@Nonnull SerialPortWrapper serialPort, byte unitIdentifier, @Nonnull Traces traces) {
            super(unitIdentifier, traces);
            this._serialPort = serialPort;
            this.setReceiveStream(new SerialPortInputStream(serialPort));
            this.setSendStream(new SerialPortOutputStream(serialPort));
            this.trace("Begin");
            this._serialPort.addStatusChangeListener(this);
        }

        @Override
        public void close() throws IOException {
            try {
                this._serialPort.removeStatusChangeListener(this);
                this._serialPort.close();
            }
            finally {
                super.close();
            }
        }

        @Override
        public int getBatchSize() {
            return 1;
        }

        @Override
        public String getLocalAddress() {
            return this._serialPort.getPortName();
        }

        @Override
        public String getRemoteAddress() {
            return this._serialPort.getPortName();
        }

        @Override
        public Prefix newPrefix() {
            return new Prefix(this.getUnitIdentifier());
        }

        @Override
        public void onMessageReceiveCompleted() throws InterruptedException {
            int portSpeed = this._serialPort.getPortSpeed();
            int millis = portSpeed <= 19200 ? (int)Math.ceil(38500.0 / (double)portSpeed - 0.1) : 2;
            Thread.sleep(millis);
        }

        @Override
        public void onMessageSendBegins() throws IOException {
            this._serialPort.purge();
        }

        @Override
        public void onStatusChange(SerialPortWrapper serialPort, SerialPortWrapper.Event event) {
            String message;
            Require.equal((Object)this._serialPort, (Object)serialPort);
            switch (event) {
                case DSR_OFF: {
                    message = "DSR off";
                    break;
                }
                case FRAME: {
                    message = "Framing error";
                    break;
                }
                case OVERRUN: {
                    message = "Overrun";
                    break;
                }
                case PARITY: {
                    message = "Parity error";
                    break;
                }
                default: {
                    message = null;
                }
            }
            if (message != null) {
                this.trace(message);
                try {
                    this.close();
                }
                catch (IOException exception) {
                    throw new InternalError(exception);
                }
            }
        }

        @Override
        public void setBatchSize(int batchSize) {
        }
    }

    private static final class _OnSerialASCIIPort
    extends _OnSerialPort {
        private static final byte _FIRST_END_BYTE = 13;
        private static final byte _SECOND_END_BYTE = 10;
        private static final byte _START_BYTE = 58;
        private final LRC _inputLRC = new LRC();
        private final LRC _outputLRC = new LRC();

        _OnSerialASCIIPort(@Nonnull SerialPortWrapper serialPort, byte unitIdentifier, @Nonnull Traces traces) {
            super(serialPort, unitIdentifier, traces);
            Require.equal((long)7L, (long)serialPort.getPortDataBits());
        }

        @Override
        public byte receiveByte() throws EOFException, IOException {
            byte firstHalfByte = _OnSerialASCIIPort.hexDigitToHalfByte(super.receiveByte());
            byte secondHalfByte = _OnSerialASCIIPort.hexDigitToHalfByte(super.receiveByte());
            byte byteValue = (byte)(firstHalfByte << 4 | secondHalfByte);
            this._inputLRC.update(byteValue);
            return byteValue;
        }

        @Override
        public Prefix receivePrefix() throws EOFException, IOException {
            byte unitIdentifier;
            while (true) {
                if (super.receiveByte() != 58) {
                    continue;
                }
                this._inputLRC.reset();
                this.getReceivedBytes().reset();
                this.getReceivedBytes().append((byte)58);
                unitIdentifier = this.receiveByte();
                if (unitIdentifier == this.getUnitIdentifier()) break;
                this.getThisLogger().trace((Messages.Entry)ModbusMessages.IGNORED_MESSAGE_FOR_UNIT, new Object[]{String.valueOf(unitIdentifier & 0xFF)});
            }
            return new Prefix(unitIdentifier);
        }

        @Override
        public boolean receiveSuffix() throws EOFException, IOException {
            byte lrc = (byte)this._inputLRC.getValue();
            boolean success = this.receiveByte() == lrc;
            byte firstEndByte = super.receiveByte();
            byte secondEndByte = firstEndByte == 13 ? (byte)super.receiveByte() : (byte)-1;
            this.traceReceivedBytes();
            if (firstEndByte != 13 || secondEndByte != 10) {
                this.getThisLogger().warn((Messages.Entry)ModbusMessages.BAD_MESSAGE_TERMINATION, new Object[]{_OnSerialASCIIPort.byteToHexString(firstEndByte), _OnSerialASCIIPort.byteToHexString(secondEndByte)});
            }
            return success;
        }

        @Override
        public void sendByte(int byteValue) throws IOException {
            super.sendByte(_OnSerialASCIIPort.halfByteToHexDigit((byte)(byteValue >> 4)));
            super.sendByte(_OnSerialASCIIPort.halfByteToHexDigit((byte)byteValue));
            this._outputLRC.update(byteValue);
        }

        @Override
        public void sendPrefix(Prefix prefix) throws IOException {
            this.getSentBytes().reset();
            super.sendByte(58);
            this._outputLRC.reset();
            prefix.write(this);
        }

        @Override
        public void sendSuffix() throws IOException {
            this.sendByte((int)this._outputLRC.getValue());
            super.sendByte(13);
            super.sendByte(10);
            this.traceSentBytes();
        }
    }
}

